//===================================================================
// EventInformation.js（安定修正版）
//===================================================================
// イベントの頭上に文字を表示するプラグイン
// 原作：蔦森くいな（MITライセンス）
// 修正版：ChatGPT（2025）
//===================================================================

/*:
 * @plugindesc v1.1.2（安定版）イベントの頭上に文字を表示します
 * @author 蔦森くいな / 修正版：ChatGPT
 * 
 * @help
 * 【使い方】
 * イベントページの１行目に「注釈」で以下を記入：
 * 
 * info:ハロルド
 * → 頭上に「ハロルド」と表示。
 * 
 * info:ハロルド,40
 * → 文字サイズ40。
 * 
 * info:ハロルド,40
 * infoMove:50,-20
 * → 位置を右50・上20にずらす。
 * 
 * \V[n] で変数の値も表示可能。
 * 
 * ※ infoMove は info の後に記入。
 * ※ スペース区切りでもOK。
 * ※ 「：」や「　」は仕様上使えません。
 *
 * 【注意】
 * 出現条件にスイッチを設定していてもエラーが出ないよう修正済。
 *
 * 利用規約：
 * MIT License（著作権表示・ライセンスURLは残してください）
 */

function _PD_EventInfomation() {
    throw new Error('This is a static class');
}

_PD_EventInfomation.infoSprite = [];
_PD_EventInfomation.changeVariables = [];

// 初期化処理（スプライト削除）
_PD_EventInfomation.infoInitialize = function (eventId) {
    var infoSprite = this.infoSprite[eventId];
    if (infoSprite && infoSprite.parent) {
        infoSprite.parent.removeChild(infoSprite);
    }
    this.infoSprite[eventId] = null;
    this.changeVariables[eventId] = null;
};

// 注釈コマンド解析
_PD_EventInfomation.setupCommand = function (event) {
    // ★ ページがないイベントは無視（出現条件で非表示時など）
    if (!event || !event.page()) return;

    _PD_EventInfomation.infoInitialize(event._eventId);
    var list = event.page().list;
    if (!list) return;

    for (var index = 0; index < list.length; index++) {
        var cmd = list[index];
        if (cmd.code !== 108 && cmd.code !== 408) continue;

        var command = cmd.parameters[0].toLowerCase().replace(/　/g, " ").split(' ');
        for (var i = 0; i < command.length; i++) {
            var param = command[i].replace(/:/g, ',').replace(/：/g, ',').split(',');
            switch (param[0]) {
                case 'info':
                    var preText = param[1];
                    var newText = _PD_EventInfomation.convertEscapeCharacters(preText);
                    if (newText !== preText) _PD_EventInfomation.changeVariables[event._eventId] = true;
                    _PD_EventInfomation.setInformation(event._eventId, newText, parseInt(param[2]));
                    break;
                case 'infomove':
                    var infoSprite = _PD_EventInfomation.infoSprite[event._eventId];
                    if (infoSprite) {
                        infoSprite.x += parseInt(param[1]) || 0;
                        infoSprite.y += parseInt(param[2]) || 0;
                    }
                    break;
            }
        }
    }
};

// テキスト描画
_PD_EventInfomation.setInformation = function (eventId, text, fontSize) {
    var charCount = 0;
    for (var i = 0; i < text.length; i++) {
        var code = text.charCodeAt(i);
        if ((code >= 0x0 && code < 0x81) ||
            (code == 0xf8f0) ||
            (code >= 0xff61 && code < 0xffa0) ||
            (code >= 0xf8f1 && code < 0xf8f4)) {
            charCount += 1;
        } else {
            charCount += 2;
        }
    }
    fontSize = fontSize || 21;
    var infoWidth = (charCount + 1) * Math.ceil(fontSize / 2);
    var infoHeight = fontSize + 2;
    var sprite = new Sprite(new Bitmap(infoWidth, infoHeight));
    sprite.anchor = new Point(0.5, 1);
    sprite.move(0, -48);
    sprite.bitmap.fillRect(0, 0, infoWidth, infoHeight, 'rgba(0,0,0,0.5)');
    sprite.bitmap.fontSize = fontSize;
    sprite.bitmap.drawText(text, 0, 0, infoWidth, infoHeight, 'center');
    _PD_EventInfomation.infoSprite[eventId] = sprite;
    _PD_EventInfomation.addInfoSprite(eventId);
};

// 全イベントに適用
_PD_EventInfomation.addInfoSpriteAll = function () {
    if (!SceneManager._scene._spriteset) return;
    $gameMap.events().forEach(e => {
        if (e && e.page()) _PD_EventInfomation.addInfoSprite(e._eventId);
    });
};

// 個別イベントに追加
_PD_EventInfomation.addInfoSprite = function (eventId) {
    if (!SceneManager._scene._spriteset) return;
    var sprite = _PD_EventInfomation.infoSprite[eventId];
    if (!(sprite instanceof Sprite)) return;

    SceneManager._scene._spriteset._characterSprites.forEach(charSprite => {
        if (charSprite._character === $gameMap._events[eventId]) {
            charSprite.addChild(sprite);
        }
    });
};

// 全削除
_PD_EventInfomation.removeInfoSpriteAll = function () {
    if (!SceneManager._scene._spriteset) return;
    $gameMap.events().forEach(e => {
        if (e) _PD_EventInfomation.removeInfoSprite(e._eventId);
    });
};

_PD_EventInfomation.removeInfoSprite = function (eventId) {
    if (!SceneManager._scene._spriteset) return;
    var sprite = _PD_EventInfomation.infoSprite[eventId];
    if (!(sprite instanceof Sprite)) return;

    SceneManager._scene._spriteset._characterSprites.forEach(charSprite => {
        if (charSprite._character === $gameMap._events[eventId]) {
            charSprite.removeChild(sprite);
        }
    });
};

// \V[n]対応
_PD_EventInfomation.convertEscapeCharacters = function (text) {
    if (!text) return '';
    text = text.replace(/\\/g, '\x1b');
    text = text.replace(/\x1b\x1b/g, '\\');
    text = text.replace(/\x1bV\[(\d+)\]/gi, (_, n) => $gameVariables.value(parseInt(n)));
    return text;
};

//===================================================================
// 主要フック
//===================================================================

(function () {
    'use strict';

    // マップ読み込み時に初期化
    const _setupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function () {
        _PD_EventInfomation.infoSprite = [];
        _PD_EventInfomation.changeVariables = [];
        _setupEvents.call(this);
    };

    // イベントページリセット時
    const _clearPageSettings = Game_Event.prototype.clearPageSettings;
    Game_Event.prototype.clearPageSettings = function () {
        _clearPageSettings.call(this);
        _PD_EventInfomation.infoInitialize(this._eventId);
    };

    // ページ設定時
    const _setupPageSettings = Game_Event.prototype.setupPageSettings;
    Game_Event.prototype.setupPageSettings = function () {
        _setupPageSettings.call(this);
        _PD_EventInfomation.setupCommand(this);
    };

    // マップ作成時（安全化済）
    const _createSpriteset = Scene_Map.prototype.createSpriteset;
    Scene_Map.prototype.createSpriteset = function () {
        _createSpriteset.call(this);
        _PD_EventInfomation.removeInfoSpriteAll();
        _PD_EventInfomation.infoSprite = [];
        _PD_EventInfomation.changeVariables = [];
        $gameMap.events().forEach(event => {
            if (event && event.page()) _PD_EventInfomation.setupCommand(event);
        });
        _PD_EventInfomation.addInfoSpriteAll();
    };

    // イベント更新時（変数反映）
    const _refresh = Game_Event.prototype.refresh;
    Game_Event.prototype.refresh = function () {
        _refresh.call(this);
        if (_PD_EventInfomation.changeVariables[this._eventId]) {
            _PD_EventInfomation.setupCommand(this);
        }
    };
})();
